"""Represents a turret"""

import math

import serge.blocks.actors

from theme import G, theme
import common 
import bullet

T_IDLE = 0
T_ACQUIRING = 1
T_LOCKED_ON = 2
T_PREPARING = 3

class Turret(serge.blocks.actors.ScreenActor):
    """Represents a turret"""

    paused = False

    def __init__(self, turret_type, board, (x, y)):
        """Initialise the Turret"""
        super(Turret, self).__init__('turret', turret_type)
        #
        self.board = board
        #
        self.name = G('name', turret_type)
        self.description = G('description', turret_type)
        self.rnge = G('range', turret_type)
        self.speed = G('speed', turret_type)
        self.damage = G('damage', turret_type)
        self.cost = G('cost', turret_type)
        self.rotation_speed = G('rotation-speed', turret_type)
        self.bullet_sprite = G('bullet-sprite', turret_type)
        self.bullet_speed = G('bullet-speed', turret_type)
        self.firing_accuracy = G('firing-accuracy', turret_type)
        self.impact_radius = G('bullet-impact-radius', turret_type)
        self.slowdown = G('slowdown', turret_type)
        #
        self.setSpriteName(turret_type)
        self.setLayerName('turrets')
        #
        self.moveTo(x, y)
        #
        self.target = None
        self.fire_timer = 0
        
    def updateActor(self, interval, world):
        """Update the turret"""
        super(Turret, self).updateActor(interval, world)
        #
        if not self.paused:
            #
            # Select target
            if not self.target or not world.hasActor(self.target):
                targets = self.board.getTargetsFor((self.x, self.y), self.rnge*2)
                if targets and targets[0][0] < self.rnge:
                    self.target = targets[0][1]
                    self.log.info('%s acquired target %s' % (self.getNiceName(), self.target.getNiceName()))
                    self.setState(T_ACQUIRING)
                elif targets:
                    self.setState(T_PREPARING)
                else:
                    self.setState(T_IDLE)
            #
            # Move towards the item of iterest
            if self.getState() != T_IDLE:
                if self.getState() == T_PREPARING:
                    the_target = targets[0][1]
                else:
                    the_target = self.target
                #
                # Calculate how to rotate the turret to face the target
                offset = common.Vec2d(the_target.x, the_target.y) - common.Vec2d(self.x, self.y)
                me = common.Vec2d.unit()
                me.angle_degrees = -self.angle
                to_go = offset.get_angle_degrees_between(me)
                #
                # Make sure that the target is still in range
                if self.getState() != T_PREPARING and offset.length > self.rnge:
                    self.target = None
                    self.log.info('Target out of range for %s' % self.getNiceName())
                    self.setState(T_IDLE)
                else:
                    change = math.copysign(min(abs(to_go), self.rotation_speed*interval/1000.0), to_go)
                    self.setAngle(self.angle + change)
                    #
                    # Allow for firing
                    if self.getState() != T_PREPARING:
                        if abs(to_go) < self.firing_accuracy:
                            if self.fire_timer > self.speed:
                                self.fire_timer = 0
                                b = bullet.Bullet((self.x, self.y), (self.target.x, self.target.y), 
                                    self.bullet_speed, self.damage, self.slowdown, self.impact_radius, 
                                        self.bullet_sprite, self.board)
                                world.addActor(b)
                            self.setState(T_LOCKED_ON)
                        else:
                            self.setState(T_ACQUIRING)
            #
            # Update firing timer
            self.fire_timer += interval/1000.0

    def setState(self, state):
        """Set the state of the turret"""
        self.visual.setCell(state)

    def getState(self):
        """Return the state"""
        return self.visual.getCell()
        
        
